jQuery(document).ready(function($) {
    const dropZone = $('#td-drop-zone');
    const fileInput = $('#td-file-input');
    const fileSelect = $('#td-file-select');
    const uploadProgress = $('#td-upload-progress');
    const progressFill = $('.td-progress-fill');
    
    // === FONCTIONNALITÉS EXISTANTES ===
    
    // Gérer le bouton de sélection de fichier
    fileSelect.on('click', function() {
        fileInput.click();
    });
    
    // Gérer la sélection de fichier
    fileInput.on('change', function() {
        const files = this.files;
        if (files.length > 0) {
            td_handleFileUpload(files[0]);
        }
    });
    
    // Gérer le drag & drop
    dropZone.on('dragover', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).addClass('td-dragover');
    });
    
    dropZone.on('dragleave', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).removeClass('td-dragover');
    });
    
    dropZone.on('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).removeClass('td-dragover');
        
        const files = e.originalEvent.dataTransfer.files;
        if (files.length > 0) {
            td_handleFileUpload(files[0]);
        }
    });
    
    // Gérer l'upload des fichiers
    function td_handleFileUpload(file, replace = false) {
        if (!file.name.toLowerCase().endsWith('.csv')) {
            alert(tdArchives.messages.invalid_file);
            return;
        }
        
        const formData = new FormData();
        formData.append('action', 'td_upload_csv');
        formData.append('nonce', tdArchives.nonce);
        formData.append('csv_file', file);
        
        if (replace) {
            formData.append('replace', '1');
        }
        
        uploadProgress.show();
        progressFill.css('width', '0%');
        dropZone.addClass('td-uploading');
        
        $.ajax({
            url: tdArchives.ajaxurl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            xhr: function() {
                const xhr = new XMLHttpRequest();
                xhr.upload.addEventListener('progress', function(e) {
                    if (e.lengthComputable) {
                        const percentComplete = (e.loaded / e.total) * 100;
                        progressFill.css('width', percentComplete + '%');
                    }
                });
                return xhr;
            },
            success: function(response) {
                uploadProgress.hide();
                dropZone.removeClass('td-uploading');
                
                if (response.success) {
                    td_showNotice(response.data.message, 'success');
                    fileInput.val('');
                    
                    setTimeout(function() {
                        window.location.href = response.data.redirect;
                    }, 2000);
                    
                } else if (response.status === 'exists') {
                    if (confirm(tdArchives.messages.file_exists)) {
                        td_handleFileUpload(file, true);
                    } else {
                        fileInput.val('');
                    }
                } else {
                    td_showNotice(response.data || tdArchives.messages.upload_error, 'error');
                }
            },
            error: function() {
                uploadProgress.hide();
                dropZone.removeClass('td-uploading');
                td_showNotice(tdArchives.messages.upload_error, 'error');
            }
        });
    }
    
    // Supprimer un fichier CSV
    $(document).on('click', '.td-delete-csv', function() {
        const filename = $(this).data('file');
        const redirect = $(this).data('redirect');
        
        if (!confirm(tdArchives.messages.delete_confirm)) {
            return;
        }
        
        const $button = $(this);
        $button.prop('disabled', true).text('Suppression...');
        
        $.ajax({
            url: tdArchives.ajaxurl,
            type: 'POST',
            data: {
                action: 'td_delete_csv',
                nonce: tdArchives.nonce,
                filename: filename,
                redirect: redirect
            },
            success: function(response) {
                if (response.success) {
                    td_showNotice(response.data.message, 'success');
                    
                    if (response.data.redirect) {
                        setTimeout(function() {
                            window.location.href = response.data.redirect;
                        }, 1500);
                    } else {
                        $button.closest('.td-file-card').fadeOut(function() {
                            $(this).remove();
                        });
                    }
                } else {
                    $button.prop('disabled', false).text('Supprimer');
                    td_showNotice(response.data || 'Erreur lors de la suppression', 'error');
                }
            },
            error: function() {
                $button.prop('disabled', false).text('Supprimer');
                td_showNotice('Erreur lors de la suppression', 'error');
            }
        });
    });
    
    // === NOUVELLES FONCTIONNALITÉS V2 ===
    
    // Renommer un fichier CSV
    $(document).on('click', '.td-rename-btn', function() {
        const $card = $(this).closest('.td-file-card');
        const $display = $card.find('.td-filename-display');
        const $input = $card.find('.td-filename-edit');
        const $button = $(this);
        
        if ($input.is(':visible')) {
            // Mode sauvegarde
            const newFilename = $input.val().trim();
            const oldFilename = $card.data('filename');
            
            if (!newFilename) {
                td_showNotice('Le nom de fichier ne peut pas être vide', 'error');
                return;
            }
            
            $button.prop('disabled', true);
            
            $.ajax({
                url: tdArchives.ajaxurl,
                type: 'POST',
                data: {
                    action: 'td_rename_csv',
                    nonce: tdArchives.nonce,
                    old_filename: oldFilename,
                    new_filename: newFilename
                },
                success: function(response) {
                    if (response.success) {
                        const fullNewFilename = response.data.new_filename;
                        $display.text(fullNewFilename);
                        $card.data('filename', fullNewFilename);
                        
                        // Mettre à jour les liens
                        $card.find('a[href*="file="]').each(function() {
                            const href = $(this).attr('href');
                            $(this).attr('href', href.replace(/file=[^&]+/, 'file=' + encodeURIComponent(fullNewFilename)));
                        });
                        
                        $card.find('.td-delete-csv').attr('data-file', fullNewFilename);
                        
                        td_showNotice(response.data.message, 'success');
                        td_exitRenameMode($card);
                    } else {
                        td_showNotice(response.data || tdArchives.messages.rename_error, 'error');
                    }
                    $button.prop('disabled', false);
                },
                error: function() {
                    td_showNotice(tdArchives.messages.rename_error, 'error');
                    $button.prop('disabled', false);
                }
            });
        } else {
            // Mode édition
            td_enterRenameMode($card);
        }
    });
    
    // Annuler le renommage avec Escape
    $(document).on('keydown', '.td-filename-edit', function(e) {
        if (e.key === 'Escape') {
            const $card = $(this).closest('.td-file-card');
            td_exitRenameMode($card);
        } else if (e.key === 'Enter') {
            const $card = $(this).closest('.td-file-card');
            $card.find('.td-rename-btn').click();
        }
    });
    
    // Fonctions helper pour le renommage
    function td_enterRenameMode($card) {
        const $display = $card.find('.td-filename-display');
        const $input = $card.find('.td-filename-edit');
        const $button = $card.find('.td-rename-btn');
        
        $display.hide();
        $input.show().focus().select();
        $button.find('.dashicons').removeClass('dashicons-edit').addClass('dashicons-yes');
        $button.attr('title', 'Sauvegarder');
    }
    
    function td_exitRenameMode($card) {
        const $display = $card.find('.td-filename-display');
        const $input = $card.find('.td-filename-edit');
        const $button = $card.find('.td-rename-btn');
        
        $input.hide();
        $display.show();
        $button.find('.dashicons').removeClass('dashicons-yes').addClass('dashicons-edit');
        $button.attr('title', 'Renommer le fichier');
        
        // Remettre la valeur originale
        const originalFilename = $card.data('filename');
        const filenameWithoutExt = originalFilename.replace(/\.csv$/i, '');
        $input.val(filenameWithoutExt);
    }
    
    // === CONVERTISSEUR HTML → CSV ===
    
    // Génération automatique du nom de fichier
    $('#td_event_name, #td_event_date').on('input change', function() {
        const eventName = $('#td_event_name').val().trim();
        const eventDate = $('#td_event_date').val();
        
        if (eventName && !$('#td_output_filename').val()) {
            let filename = 'participants_' + td_slugify(eventName);
            if (eventDate) {
                const date = new Date(eventDate);
                const year = date.getFullYear();
                filename += '_' + year;
            }
            filename += '.csv';
            $('#td_output_filename').val(filename);
        }
    });
    
    // Prévisualisation de la conversion
    $('#td_preview_conversion').on('click', function() {
        const formData = td_getConverterFormData();
        if (!formData) return;
        
        formData.append('preview_only', 'true');
        
        const $button = $(this);
        const originalText = $button.text();
        $button.prop('disabled', true).text('Analyse en cours...');
        
        $.ajax({
            url: tdArchives.ajaxurl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    $('#td_preview_content').html(response.data.preview);
                    $('#td_conversion_preview').show();
                    td_showNotice('Analyse terminée', 'success');
                } else {
                    td_showNotice(response.data || tdArchives.messages.conversion_error, 'error');
                }
                $button.prop('disabled', false).text(originalText);
            },
            error: function() {
                td_showNotice(tdArchives.messages.conversion_error, 'error');
                $button.prop('disabled', false).text(originalText);
            }
        });
    });
    
    // Conversion et sauvegarde
    $('#td-html-converter-form').on('submit', function(e) {
        e.preventDefault();
        
        const formData = td_getConverterFormData();
        if (!formData) return;
        
        const $submitBtn = $('#td_convert_and_save');
        const originalText = $submitBtn.text();
        $submitBtn.prop('disabled', true).text('Conversion en cours...');
        
        $.ajax({
            url: tdArchives.ajaxurl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    td_showNotice(response.data.message + ' (' + response.data.count + ' participants)', 'success');
                    
                    // Rediriger vers la liste après succès
                    setTimeout(function() {
                        window.location.href = response.data.redirect;
                    }, 2000);
                } else {
                    td_showNotice(response.data || tdArchives.messages.conversion_error, 'error');
                    $submitBtn.prop('disabled', false).text(originalText);
                }
            },
            error: function() {
                td_showNotice(tdArchives.messages.conversion_error, 'error');
                $submitBtn.prop('disabled', false).text(originalText);
            }
        });
    });
    
    // Fonction helper pour récupérer les données du formulaire de conversion
    function td_getConverterFormData() {
        const eventName = $('#td_event_name').val().trim();
        const eventDate = $('#td_event_date').val();
        const eventLocation = $('#td_event_location').val().trim();
        const htmlContent = $('#td_html_content').val().trim();
        const outputFilename = $('#td_output_filename').val().trim();
        
        if (!eventName || !eventDate || !eventLocation || !htmlContent) {
            td_showNotice(tdArchives.messages.missing_fields, 'error');
            return null;
        }
        
        const formData = new FormData();
        formData.append('action', 'td_convert_html_to_csv');
        formData.append('nonce', tdArchives.nonce);
        formData.append('event_name', eventName);
        formData.append('event_date', eventDate);
        formData.append('event_location', eventLocation);
        formData.append('html_content', htmlContent);
        formData.append('output_filename', outputFilename);
        
        return formData;
    }
    
    // Fonction helper pour créer un slug
    function td_slugify(text) {
        return text
            .toLowerCase()
            .normalize('NFD')
            .replace(/[\u0300-\u036f]/g, '')
            .replace(/[^a-z0-9\s-]/g, '')
            .replace(/\s+/g, '_')
            .replace(/_+/g, '_')
            .replace(/^_|_$/g, '');
    }
    
    // === NOUVELLES FONCTIONNALITÉS DE RECHERCHE (V2 Enhanced) ===
    
    // Variables pour la recherche
    let searchTimeout;
    const searchDelay = 300; // délai en millisecondes
    
    // Recherche AJAX en temps réel
    $('#td-search-input').on('input', function() {
        const searchQuery = $(this).val().trim();
        
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(function() {
            if (searchQuery.length >= 2 || searchQuery.length === 0) {
                td_performAjaxSearch(searchQuery);
            }
        }, searchDelay);
    });
    
    // Fonction de recherche AJAX
    function td_performAjaxSearch(query) {
        // Note: Cette fonction pourrait être implémentée pour une recherche AJAX en temps réel
        // Pour l'instant, nous utilisons la recherche par formulaire standard
        // qui est plus fiable et compatible avec la pagination
        
        // Si vous voulez une recherche AJAX instantanée, décommentez le code suivant :
        /*
        $.ajax({
            url: tdArchives.ajaxurl,
            type: 'POST',
            data: {
                action: 'td_search_csv_files',
                nonce: tdArchives.nonce,
                search_query: query
            },
            success: function(response) {
                if (response.success) {
                    td_displaySearchResults(response.data, query);
                }
            },
            error: function() {
                console.log('Erreur lors de la recherche');
            }
        });
        */
    }
    
    // Afficher les résultats de recherche AJAX (optionnel)
    function td_displaySearchResults(results, query) {
        let html = '';
        
        if (results.length === 0) {
            html = '<div class="td-empty-state">';
            html += '<p>Aucun fichier trouvé pour "' + query + '"</p>';
            html += '<div class="td-empty-actions">';
            html += '<a href="' + window.location.pathname + '?page=td-archives-list" class="button button-secondary">Voir tous les fichiers</a>';
            html += '</div></div>';
        } else {
            results.forEach(function(file) {
                html += '<div class="td-file-card">';
                html += '<div class="td-file-header">';
                html += '<h3 class="td-file-title">';
                html += '<span class="td-filename-display">' + file.filename + '</span>';
                if (file.event_date) {
                    html += '<span class="td-event-date-badge">' + file.event_date + '</span>';
                }
                html += '</h3>';
                html += '<div class="td-file-actions">';
                html += '<a href="' + file.url + '" class="button button-primary">Voir le détail</a>';
                html += '</div></div></div>';
            });
        }
        
        $('.td-list-container').html(html);
        
        // Afficher le nombre de résultats
        $('.td-search-results').remove();
        if (query) {
            const resultHtml = '<div class="td-search-results">';
            resultHtml += '<p><strong>' + results.length + '</strong> résultat(s) pour "<em>' + query + '</em>"</p>';
            resultHtml += '</div>';
            $('.td-search-controls').after(resultHtml);
        }
    }
    
    // Auto-submit du formulaire de recherche/tri quand on change les selects
    $('.td-sort-select, .td-order-select').on('change', function() {
        // Auto-submit pour une meilleure UX
        // $(this).closest('form').submit();
    });
    
    // Smooth scroll après changement de page
    if (window.location.href.indexOf('paged=') > -1 || 
        window.location.href.indexOf('s=') > -1 || 
        window.location.href.indexOf('sort=') > -1) {
        setTimeout(function() {
            $('html, body').animate({
                scrollTop: $('.wrap').offset().top - 32
            }, 300);
        }, 100);
    }
    
    // Highlight des termes de recherche dans les résultats
    const urlParams = new URLSearchParams(window.location.search);
    const searchTerm = urlParams.get('s');
    if (searchTerm) {
        $('.td-filename-display').each(function() {
            const text = $(this).text();
            const regex = new RegExp('(' + searchTerm.replace(/[.*+?^${}()|[\]\\]/g, '\\$&') + ')', 'gi');
            const highlighted = text.replace(regex, '<mark>$1</mark>');
            $(this).html(highlighted);
        });
    }
    
    // Animation d'entrée pour les cartes de fichiers
    $('.td-file-card').each(function(index) {
        $(this).css({
            'opacity': '0',
            'transform': 'translateY(20px)'
        });
        
        setTimeout(() => {
            $(this).animate({
                'opacity': '1',
                'transform': 'translateY(0)'
            }, 300);
        }, index * 50);
    });
    
    // Gestion du focus sur la barre de recherche avec raccourci clavier
    $(document).on('keydown', function(e) {
        // Ctrl/Cmd + K pour focus sur la recherche
        if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
            e.preventDefault();
            $('#td-search-input').focus();
        }
        
        // Escape pour vider la recherche
        if (e.key === 'Escape' && $('#td-search-input').is(':focus')) {
            $('#td-search-input').val('');
            if (window.location.search.includes('s=')) {
                window.location.href = window.location.pathname + '?page=td-archives-list';
            }
        }
    });
    
    // Amélioration de l'UX pour les états de chargement
    $('.td-search-form').on('submit', function() {
        const $submitBtn = $(this).find('button[type="submit"]');
        const originalText = $submitBtn.html();
        $submitBtn.prop('disabled', true).html('<span class="dashicons dashicons-update-alt" style="animation: spin 1s infinite linear;"></span> Recherche...');
        
        // Remettre le bouton normal après un délai
        setTimeout(function() {
            $submitBtn.prop('disabled', false).html(originalText);
        }, 2000);
    });
    
    // Fonction pour afficher/masquer les filtres avancés (si ajoutés plus tard)
    function td_toggleAdvancedFilters() {
        const $advanced = $('.td-advanced-filters');
        if ($advanced.length) {
            $advanced.slideToggle(300);
        }
    }
    
    // Sauvegarde automatique des préférences de tri dans localStorage
    $('.td-sort-select, .td-order-select').on('change', function() {
        const sortBy = $('.td-sort-select').val();
        const sortOrder = $('.td-order-select').val();
        
        if (localStorage) {
            localStorage.setItem('td_archives_sort_by', sortBy);
            localStorage.setItem('td_archives_sort_order', sortOrder);
        }
    });
    
    // Restaurer les préférences de tri au chargement
    if (localStorage && !window.location.search.includes('sort=')) {
        const savedSortBy = localStorage.getItem('td_archives_sort_by');
        const savedSortOrder = localStorage.getItem('td_archives_sort_order');
        
        if (savedSortBy) {
            $('.td-sort-select').val(savedSortBy);
        }
        if (savedSortOrder) {
            $('.td-order-select').val(savedSortOrder);
        }
    }
    
    // === FONCTIONNALITÉS EXISTANTES (suite) ===
    
    // Afficher une notification
    function td_showNotice(message, type) {
        $('.notice').remove();
        
        const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
        const notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p><button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss this notice.</span></button></div>');
        
        $('.wrap h1').after(notice);
        
        setTimeout(function() {
            notice.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
        
        notice.on('click', '.notice-dismiss', function() {
            notice.remove();
        });
    }
    
    // Animation des cartes de fichiers au hover
    $('.td-file-card').on('mouseenter', function() {
        $(this).addClass('td-hover');
    }).on('mouseleave', function() {
        $(this).removeClass('td-hover');
    });
    
    // Scroll vers le haut après pagination
    if (window.location.href.indexOf('paged=') > -1) {
        $('html, body').animate({
            scrollTop: $('.wrap').offset().top - 32
        }, 500);
    }

    // Gestion des clics sur les tags de colonnes (page options)
    $(document).on('click', '.td-column-tag', function() {
        const columnName = $(this).text();
        const textarea = $('#td_hidden_columns');
        const currentValue = textarea.val().trim();
        
        const lines = currentValue.split('\n');
        const isAlreadyAdded = lines.some(line => {
            const columns = line.split(',').map(col => col.trim());
            return columns.includes(columnName);
        });
        
        if (!isAlreadyAdded) {
            const newValue = currentValue ? currentValue + '\n' + columnName : columnName;
            textarea.val(newValue);
            
            $(this).addClass('td-tag-added');
            setTimeout(() => {
                $(this).removeClass('td-tag-added');
            }, 1000);
            
            td_showNotice('Colonne "' + columnName + '" ajoutée à la liste', 'success');
        } else {
            td_showNotice('Cette colonne est déjà dans la liste', 'error');
        }
    });

});
