document.addEventListener("DOMContentLoaded", function() {
    initCarousels();
});

function initCarousels() {
    const carousels = document.querySelectorAll(".td-lab-carousel");
    
    if (!carousels.length) {
        return;
    }
    
    carousels.forEach(function(carousel) {
        const track = carousel.querySelector(".td-lab-track");
        const prevBtn = carousel.querySelector(".td-lab-prev");
        const nextBtn = carousel.querySelector(".td-lab-next");
        const slides = carousel.querySelectorAll(".td-lab-slide");
        
        if (!track || !prevBtn || !nextBtn || !slides.length) {
            return;
        }
        
        let currentIndex = 0;
        let autoPlayInterval;
        const autoPlayDelay = 3000;
        
        function getSlidesPerView() {
            return window.innerWidth < 1000 ? 1 : 3;
        }
        
        function getMaxIndex() {
            const slidesPerView = getSlidesPerView();
            return Math.max(0, slides.length - slidesPerView);
        }
        
        function updateCarousel() {
            const slideWidth = slides[0].offsetWidth;
            const trackStyle = window.getComputedStyle(track);
            const gap = parseFloat(trackStyle.gap) || 24;
            const offset = -(currentIndex * (slideWidth + gap));
            track.style.transform = `translateX(${offset}px)`;
            
            const maxIndex = getMaxIndex();
            
            if (maxIndex === 0) {
                prevBtn.style.display = "none";
                nextBtn.style.display = "none";
                return;
            }
            
            prevBtn.style.display = "flex";
            nextBtn.style.display = "flex";
            
            prevBtn.disabled = currentIndex === 0;
            nextBtn.disabled = currentIndex >= maxIndex;
            
            prevBtn.style.opacity = currentIndex === 0 ? "0.3" : "1";
            prevBtn.style.cursor = currentIndex === 0 ? "not-allowed" : "pointer";
            nextBtn.style.opacity = currentIndex >= maxIndex ? "0.3" : "1";
            nextBtn.style.cursor = currentIndex >= maxIndex ? "not-allowed" : "pointer";
        }
        
        function goToNext() {
            const maxIndex = getMaxIndex();
            if (currentIndex < maxIndex) {
                currentIndex++;
            } else {
                currentIndex = 0;
            }
            updateCarousel();
        }
        
        function goToPrev() {
            if (currentIndex > 0) {
                currentIndex--;
            } else {
                currentIndex = getMaxIndex();
            }
            updateCarousel();
        }
        
        function startAutoPlay() {
            stopAutoPlay();
            if (getMaxIndex() > 0) {
                autoPlayInterval = setInterval(goToNext, autoPlayDelay);
            }
        }
        
        function stopAutoPlay() {
            if (autoPlayInterval) {
                clearInterval(autoPlayInterval);
            }
        }
        
        prevBtn.addEventListener("click", function(e) {
            e.preventDefault();
            goToPrev();
            stopAutoPlay();
            setTimeout(startAutoPlay, 5000);
        });
        
        nextBtn.addEventListener("click", function(e) {
            e.preventDefault();
            goToNext();
            stopAutoPlay();
            setTimeout(startAutoPlay, 5000);
        });
        
        carousel.addEventListener("mouseenter", stopAutoPlay);
        carousel.addEventListener("mouseleave", startAutoPlay);
        
        let resizeTimer;
        window.addEventListener("resize", function() {
            clearTimeout(resizeTimer);
            resizeTimer = setTimeout(function() {
                currentIndex = Math.min(currentIndex, getMaxIndex());
                updateCarousel();
            }, 250);
        });
        
        updateCarousel();
        startAutoPlay();
    });
}

if (window.wp && window.wp.domReady) {
    window.wp.domReady(initCarousels);
}
