(function(wp) {
    const { registerBlockType } = wp.blocks;
    const { useBlockProps } = wp.blockEditor;
    const { useEffect, useState, createElement: el } = wp.element;
    
    registerBlockType("td/laboratoires-liste", {
        title: "Liste Laboratoires",
        icon: "list-view",
        category: "widgets",
        description: "Affiche la liste complète des laboratoires avec leurs informations",
        
        edit: function() {
            const blockProps = useBlockProps();
            const [laboratoires, setLaboratoires] = useState([]);
            const [loading, setLoading] = useState(true);
            
            useEffect(() => {
                wp.apiFetch({ path: "/wp/v2/td_laboratoire?per_page=100&status=publish" })
                    .then((data) => {
                        if (!data || data.length === 0) {
                            setLaboratoires([]);
                            setLoading(false);
                            return;
                        }
                        
                        const labsPromises = data.map(lab => {
                            const labData = {
                                id: lab.id,
                                title: lab.title.rendered,
                                ville: lab.meta._td_lab_ville || "",
                                description: lab.meta._td_lab_description || "",
                                lien: lab.meta._td_lab_lien || "",
                                contact: lab.meta._td_lab_contact || "",
                                logo: lab.featured_media
                            };
                            
                            if (!labData.logo) return Promise.resolve(labData);
                            
                            return wp.apiFetch({ path: `/wp/v2/media/${labData.logo}` })
                                .then(media => ({
                                    ...labData,
                                    logoUrl: media.media_details?.sizes?.medium?.source_url || media.source_url
                                }))
                                .catch(() => labData);
                        });
                        
                        return Promise.all(labsPromises);
                    })
                    .then((labs) => {
                        if (labs) {
                            setLaboratoires(labs);
                        }
                        setLoading(false);
                    })
                    .catch((error) => {
                        console.error("Erreur lors du chargement des laboratoires:", error);
                        setLoading(false);
                    });
            }, []);
            
            if (loading) {
                return el(
                    "div",
                    blockProps,
                    el("p", { style: { textAlign: "center", padding: "20px" } }, "Chargement des laboratoires...")
                );
            }
            
            if (laboratoires.length === 0) {
                const adminUrl = window.location.origin + "/wp-admin/edit.php?post_type=td_laboratoire";
                return el(
                    "div",
                    blockProps,
                    el(
                        "div",
                        { style: { padding: "20px", background: "#f0f0f0", border: "2px dashed #ccc", textAlign: "center" } },
                        el("span", { className: "dashicons dashicons-list-view", style: { fontSize: "40px", color: "#666" } }),
                        el("p", { style: { margin: "10px 0 0 0", color: "#666" } }, "Aucun laboratoire ajouté"),
                        el("p", { style: { margin: "5px 0 0 0", fontSize: "12px", color: "#999" } }, "Ajoutez des laboratoires pour les voir ici"),
                        el(
                            "a",
                            { 
                                href: adminUrl,
                                style: { 
                                    display: "inline-block",
                                    marginTop: "10px",
                                    padding: "8px 16px",
                                    background: "#0073aa",
                                    color: "#fff",
                                    textDecoration: "none",
                                    borderRadius: "4px",
                                    fontSize: "13px"
                                }
                            },
                            "Gérer les laboratoires"
                        )
                    )
                );
            }
            
            return el(
                "div",
                blockProps,
                el(
                    "div",
                    { className: "td-lab-liste", style: { padding: "20px", background: "#f9f9f9", borderRadius: "8px" } },
                    el("p", { style: { textAlign: "center", fontSize: "12px", color: "#999", marginBottom: "15px" } }, `Aperçu de la liste (${laboratoires.length} laboratoires)`),
                    laboratoires.slice(0, 3).map(lab => 
                        el(
                            "div",
                            { 
                                key: lab.id, 
                                className: "td-lab-item", 
                                style: { 
                                    display: "flex", 
                                    gap: "1.5rem", 
                                    padding: "1.5rem", 
                                    background: "#fff", 
                                    borderRadius: "8px", 
                                    boxShadow: "0 2px 8px rgba(0,0,0,0.1)",
                                    marginBottom: "1rem"
                                } 
                            },
                            lab.logoUrl ? el(
                                "div",
                                { style: { flexShrink: 0, width: "120px" } },
                                el("img", { src: lab.logoUrl, alt: lab.title, style: { width: "100%", height: "auto" } })
                            ) : null,
                            el(
                                "div",
                                { style: { flex: 1 } },
                                el("h3", { style: { fontSize: "1.3rem", margin: "0 0 0.75rem 0", color: "#333" } }, lab.title),
                                lab.ville ? el("p", { style: { margin: "0.5rem 0", fontSize: "0.95rem", color: "#555" } }, el("strong", null, "Ville : "), lab.ville) : null,
                                lab.description ? el("p", { style: { margin: "0.5rem 0", fontSize: "0.95rem", color: "#666" } }, lab.description) : null,
                                lab.contact ? el("p", { style: { margin: "0.5rem 0", fontSize: "0.95rem", color: "#555" } }, el("strong", null, "Contact : "), lab.contact) : null,
                                lab.lien ? el("p", { style: { margin: "0.75rem 0 0 0" } }, el("a", { href: lab.lien, target: "_blank", rel: "noopener noreferrer", style: { display: "inline-block", padding: "0.4rem 1rem", background: "#0073aa", color: "#fff", textDecoration: "none", borderRadius: "4px", fontSize: "0.9rem" } }, "Visiter le site")) : null
                            )
                        )
                    ),
                    laboratoires.length > 3 ? el("p", { style: { textAlign: "center", fontSize: "11px", color: "#999", marginTop: "10px" } }, `... et ${laboratoires.length - 3} autres laboratoires`) : null
                )
            );
        },
        
        save: function() {
            return null;
        }
    });
})(window.wp);
