<?php
// Bloc : Liste complète des rencontres

function td_rencontres_register_liste_block() {
    if (!function_exists('register_block_type')) {
        return;
    }
    
    register_block_type('td-rencontres/liste-rencontres', array(
        'attributes' => array(
            'categorie' => array(
                'type' => 'string',
                'default' => 'all'
            ),
            'ordre' => array(
                'type' => 'string',
                'default' => 'desc'
            )
        ),
        'render_callback' => 'td_rencontres_render_liste_block'
    ));
}
add_action('init', 'td_rencontres_register_liste_block');

function td_rencontres_render_liste_block($attributes) {
    $categorie = isset($attributes['categorie']) ? $attributes['categorie'] : 'all';
    $ordre = isset($attributes['ordre']) ? $attributes['ordre'] : 'desc';
    
    $args = array(
        'post_type' => 'rencontre',
        'posts_per_page' => -1,
        'post_status' => 'publish',
        'meta_key' => '_td_date_evenement',
        'orderby' => 'meta_value',
        'order' => $ordre
    );
    
    if ($categorie !== 'all') {
        $args['tax_query'] = array(
            array(
                'taxonomy' => 'categorie_rencontre',
                'field' => 'slug',
                'terms' => $categorie
            )
        );
    }
    
    $rencontres = new WP_Query($args);
    
    if (!$rencontres->have_posts()) {
        return '<p class="td-rencontres-vide">Aucune rencontre trouvée.</p>';
    }
    
    ob_start();
    ?>
    <div class="td-rencontres-liste">
        <div class="td-rencontres-filtres">
            <label for="td-filtre-categorie">Filtrer par catégorie :</label>
            <select id="td-filtre-categorie" class="td-filtre-select">
                <option value="all" <?php selected($categorie, 'all'); ?>>Toutes les catégories</option>
                <?php
                $categories = get_terms(array(
                    'taxonomy' => 'categorie_rencontre',
                    'hide_empty' => false
                ));
                foreach ($categories as $cat) {
                    echo '<option value="' . esc_attr($cat->slug) . '" ' . selected($categorie, $cat->slug, false) . '>' . esc_html($cat->name) . '</option>';
                }
                ?>
            </select>
            
            <label for="td-filtre-ordre">Trier par :</label>
            <select id="td-filtre-ordre" class="td-filtre-select">
                <option value="desc" <?php selected($ordre, 'desc'); ?>>Plus récentes en premier</option>
                <option value="asc" <?php selected($ordre, 'asc'); ?>>Plus anciennes en premier</option>
            </select>
        </div>
        
        <div class="td-rencontres-grille">
            <?php
            while ($rencontres->have_posts()) {
                $rencontres->the_post();
                $post_id = get_the_ID();
                $date_evenement = get_post_meta($post_id, '_td_date_evenement', true);
                $lieu = get_post_meta($post_id, '_td_lieu', true);
                $statut = get_post_meta($post_id, '_td_statut', true);
                $responsable = get_post_meta($post_id, '_td_responsable', true);
                $nb_participants = get_post_meta($post_id, '_td_nb_participants', true);
                $url_sciencesconf = get_post_meta($post_id, '_td_url_sciencesconf', true);
                $terms = get_the_terms($post_id, 'categorie_rencontre');
                $categorie_name = $terms && !is_wp_error($terms) ? $terms[0]->name : '';
                ?>
                <article class="td-rencontre-item <?php echo esc_attr($statut); ?>">
                    <?php if (has_post_thumbnail()) : ?>
                        <div class="td-rencontre-image">
                            <?php the_post_thumbnail('medium'); ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="td-rencontre-content">
                        <div class="td-rencontre-meta-top">
                            <?php if ($categorie_name) : ?>
                                <span class="td-rencontre-categorie"><?php echo esc_html($categorie_name); ?></span>
                            <?php endif; ?>
                            <?php if ($statut) : ?>
                                <span class="td-rencontre-statut <?php echo esc_attr($statut); ?>">
                                    <?php echo $statut === 'passee' ? 'Passée' : 'À venir'; ?>
                                </span>
                            <?php endif; ?>
                        </div>
                        
                        <h3 class="td-rencontre-title">
                            <a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
                        </h3>
                        
                        <?php if ($date_evenement) : ?>
                            <p class="td-rencontre-date">
                                <strong>📅 Date :</strong> <?php echo date_i18n('d/m/Y', strtotime($date_evenement)); ?>
                            </p>
                        <?php endif; ?>
                        
                        <?php if ($lieu) : ?>
                            <p class="td-rencontre-lieu">
                                <strong>📍 Lieu :</strong> <?php echo esc_html($lieu); ?>
                            </p>
                        <?php endif; ?>
                        
                        <?php if ($responsable) : ?>
                            <p class="td-rencontre-responsable">
                                <strong>👤 Responsable :</strong> <?php echo esc_html($responsable); ?>
                            </p>
                        <?php endif; ?>
                        
                        <?php if ($nb_participants) : ?>
                            <p class="td-rencontre-participants">
                                <strong>👥 Participants :</strong> <?php echo esc_html($nb_participants); ?>
                            </p>
                        <?php endif; ?>
                        
                        <div class="td-rencontre-actions">
                            <a href="<?php the_permalink(); ?>" class="td-btn td-btn-primary">Voir les détails</a>
                            <?php if ($url_sciencesconf) : ?>
                                <a href="<?php echo esc_url($url_sciencesconf); ?>" target="_blank" rel="noopener" class="td-btn td-btn-secondary">SciencesConf</a>
                            <?php endif; ?>
                        </div>
                    </div>
                </article>
                <?php
            }
            wp_reset_postdata();
            ?>
        </div>
    </div>
    
    <style>
        .td-rencontres-liste { margin: 20px 0; }
        
        .td-rencontres-filtres {
            background: #f5f5f5;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            align-items: center;
        }
        
        .td-rencontres-filtres label {
            font-weight: bold;
            margin: 0;
        }
        
        .td-filtre-select {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }
        
        .td-rencontres-grille {
            display: grid;
            gap: 20px;
            grid-template-columns: 1fr;
        }
        
        @media (min-width: 1000px) {
            .td-rencontres-grille {
                grid-template-columns: repeat(2, 1fr);
            }
        }
        
        .td-rencontre-item {
            border: 1px solid #ddd;
            border-radius: 8px;
            overflow: hidden;
            background: #fff;
            transition: box-shadow 0.3s;
        }
        
        .td-rencontre-item:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .td-rencontre-item.passee {
            opacity: 0.8;
        }
        
        .td-rencontre-image img {
            width: 100%;
            height: 200px;
            object-fit: cover;
        }
        
        .td-rencontre-content {
            padding: 20px;
        }
        
        .td-rencontre-meta-top {
            display: flex;
            gap: 10px;
            margin-bottom: 10px;
            flex-wrap: wrap;
        }
        
        .td-rencontre-categorie,
        .td-rencontre-statut {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .td-rencontre-categorie {
            background: #e3f2fd;
            color: #1976d2;
        }
        
        .td-rencontre-statut {
            background: #f5f5f5;
            color: #666;
        }
        
        .td-rencontre-statut.a_venir {
            background: #e8f5e9;
            color: #2e7d32;
        }
        
        .td-rencontre-title {
            font-size: 20px;
            margin: 0 0 15px 0;
            line-height: 1.3;
        }
        
        @media (max-width: 999px) {
            .td-rencontre-title {
                font-size: 18px;
            }
        }
        
        .td-rencontre-title a {
            color: #333;
            text-decoration: none;
        }
        
        .td-rencontre-title a:hover {
            color: #1976d2;
        }
        
        .td-rencontre-content p {
            margin: 8px 0;
            font-size: 14px;
            color: #555;
        }
        
        .td-rencontre-actions {
            display: flex;
            gap: 10px;
            margin-top: 15px;
            flex-wrap: wrap;
        }
        
        .td-btn {
            display: inline-block;
            padding: 10px 20px;
            text-decoration: none;
            border-radius: 4px;
            font-size: 14px;
            font-weight: bold;
            transition: all 0.3s;
            text-align: center;
        }
        
        .td-btn-primary {
            background: #1976d2;
            color: #fff;
        }
        
        .td-btn-primary:hover {
            background: #1565c0;
            color: #fff;
        }
        
        .td-btn-secondary {
            background: #fff;
            color: #1976d2;
            border: 1px solid #1976d2;
        }
        
        .td-btn-secondary:hover {
            background: #1976d2;
            color: #fff;
        }
        
        .td-rencontres-vide {
            text-align: center;
            padding: 40px;
            color: #666;
            font-style: italic;
        }
    </style>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const filtreCategorie = document.getElementById('td-filtre-categorie');
        const filtreOrdre = document.getElementById('td-filtre-ordre');
        
        if (filtreCategorie) {
            filtreCategorie.addEventListener('change', function() {
                window.location.href = window.location.pathname + '?categorie=' + this.value + '&ordre=' + filtreOrdre.value;
            });
        }
        
        if (filtreOrdre) {
            filtreOrdre.addEventListener('change', function() {
                window.location.href = window.location.pathname + '?categorie=' + filtreCategorie.value + '&ordre=' + this.value;
            });
        }
    });
    </script>
    <?php
    
    return ob_get_clean();
}
