<?php
/*
Plugin Name: TD Rencontres
Description: Gestion des rencontres scientifiques et journées thématiques
Version: 1.1.0
Author: Thomas Dufranne
Author URI: https://td-plugins.com
Text Domain: td-rencontres
*/

if (!defined('ABSPATH')) {
    exit;
}

define('TD_RENCONTRES_VERSION', '1.1.0');
define('TD_RENCONTRES_PATH', plugin_dir_path(__FILE__));
define('TD_RENCONTRES_URL', plugin_dir_url(__FILE__));

// Fonction d'activation
function td_rencontres_activate() {
    td_rencontres_register_post_type();
    td_rencontres_register_taxonomy();
    flush_rewrite_rules();
    
    // Créer les catégories par défaut pour le statut de l'événement
    if (!term_exists('À venir', 'categorie_rencontre')) {
        wp_insert_term('À venir', 'categorie_rencontre');
    }
    if (!term_exists('Passée', 'categorie_rencontre')) {
        wp_insert_term('Passée', 'categorie_rencontre');
    }
    
    // Créer les catégories par défaut pour les inscriptions
    if (!term_exists('Inscriptions ouvertes', 'categorie_rencontre')) {
        wp_insert_term('Inscriptions ouvertes', 'categorie_rencontre');
    }
    if (!term_exists('Inscriptions fermées', 'categorie_rencontre')) {
        wp_insert_term('Inscriptions fermées', 'categorie_rencontre');
    }
}
register_activation_hook(__FILE__, 'td_rencontres_activate');

// Fonction de désactivation
function td_rencontres_deactivate() {
    flush_rewrite_rules();
}
register_deactivation_hook(__FILE__, 'td_rencontres_deactivate');

// Enregistrer le Custom Post Type
function td_rencontres_register_post_type() {
    $labels = array(
        'name' => 'Rencontres',
        'singular_name' => 'Rencontre',
        'menu_name' => 'Rencontres',
        'add_new' => 'Ajouter une rencontre',
        'add_new_item' => 'Ajouter une nouvelle rencontre',
        'edit_item' => 'Modifier la rencontre',
        'new_item' => 'Nouvelle rencontre',
        'view_item' => 'Voir la rencontre',
        'search_items' => 'Rechercher des rencontres',
        'not_found' => 'Aucune rencontre trouvée',
        'not_found_in_trash' => 'Aucune rencontre dans la corbeille'
    );

    $args = array(
        'labels' => $labels,
        'public' => true,
        'has_archive' => true,
        'publicly_queryable' => true,
        'show_ui' => true,
        'show_in_menu' => true,
        'show_in_rest' => true,
        'menu_icon' => 'dashicons-groups',
        'supports' => array('title', 'editor', 'thumbnail'),
        'rewrite' => array('slug' => 'rencontres'),
        'capability_type' => 'post'
    );

    register_post_type('rencontre', $args);
}
add_action('init', 'td_rencontres_register_post_type');

// Enregistrer la taxonomie
function td_rencontres_register_taxonomy() {
    $labels = array(
        'name' => 'Catégories de rencontre',
        'singular_name' => 'Catégorie de rencontre',
        'search_items' => 'Rechercher des catégories',
        'all_items' => 'Toutes les catégories',
        'edit_item' => 'Modifier la catégorie',
        'update_item' => 'Mettre à jour la catégorie',
        'add_new_item' => 'Ajouter une nouvelle catégorie',
        'new_item_name' => 'Nom de la nouvelle catégorie',
        'menu_name' => 'Catégories'
    );

    $args = array(
        'labels' => $labels,
        'hierarchical' => true,
        'public' => true,
        'show_ui' => true,
        'show_in_rest' => true,
        'show_admin_column' => true,
        'rewrite' => array('slug' => 'categorie-rencontre')
    );

    register_taxonomy('categorie_rencontre', array('rencontre'), $args);
}
add_action('init', 'td_rencontres_register_taxonomy');

// Ajouter les meta boxes
function td_rencontres_add_meta_boxes() {
    add_meta_box(
        'td_rencontres_details',
        'Détails de la rencontre',
        'td_rencontres_meta_box_callback',
        'rencontre',
        'normal',
        'high'
    );
}
add_action('add_meta_boxes', 'td_rencontres_add_meta_boxes');

// Callback pour afficher la meta box
function td_rencontres_meta_box_callback($post) {
    wp_nonce_field('td_rencontres_save_meta', 'td_rencontres_nonce');
    
    $date_evenement = get_post_meta($post->ID, '_td_date_evenement', true);
    $date_ouverture_inscriptions = get_post_meta($post->ID, '_td_date_ouverture_inscriptions', true);
    $date_fermeture_inscriptions = get_post_meta($post->ID, '_td_date_fermeture_inscriptions', true);
    $lieu = get_post_meta($post->ID, '_td_lieu', true);
    $url_sciencesconf = get_post_meta($post->ID, '_td_url_sciencesconf', true);
    $nb_participants = get_post_meta($post->ID, '_td_nb_participants', true);
    $nb_laboratoires = get_post_meta($post->ID, '_td_nb_laboratoires', true);
    
    ?>
    <style>
        .td-rencontres-meta-field { margin-bottom: 15px; }
        .td-rencontres-meta-field label { display: block; font-weight: bold; margin-bottom: 5px; }
        .td-rencontres-meta-field input[type="text"],
        .td-rencontres-meta-field input[type="date"],
        .td-rencontres-meta-field input[type="url"],
        .td-rencontres-meta-field input[type="number"] { width: 100%; max-width: 400px; }
    </style>
    
    <div class="td-rencontres-meta-field">
        <label for="td_date_evenement">Date de l'événement *</label>
        <input type="date" id="td_date_evenement" name="td_date_evenement" value="<?php echo esc_attr($date_evenement); ?>" required>
    </div>
    
    <div class="td-rencontres-meta-field">
        <label for="td_date_ouverture_inscriptions">Date d'ouverture des inscriptions</label>
        <input type="date" id="td_date_ouverture_inscriptions" name="td_date_ouverture_inscriptions" value="<?php echo esc_attr($date_ouverture_inscriptions); ?>">
    </div>
    
    <div class="td-rencontres-meta-field">
        <label for="td_date_fermeture_inscriptions">Date de fermeture des inscriptions</label>
        <input type="date" id="td_date_fermeture_inscriptions" name="td_date_fermeture_inscriptions" value="<?php echo esc_attr($date_fermeture_inscriptions); ?>">
    </div>
    
    <div class="td-rencontres-meta-field">
        <label for="td_lieu">Lieu</label>
        <input type="text" id="td_lieu" name="td_lieu" value="<?php echo esc_attr($lieu); ?>">
    </div>
    
    <div class="td-rencontres-meta-field">
        <label for="td_url_sciencesconf">URL (Page sciencesconf.org)</label>
        <input type="url" id="td_url_sciencesconf" name="td_url_sciencesconf" value="<?php echo esc_url($url_sciencesconf); ?>">
    </div>
    
    <div class="td-rencontres-meta-field">
        <label for="td_nb_participants">Nombre de participants</label>
        <input type="number" id="td_nb_participants" name="td_nb_participants" value="<?php echo esc_attr($nb_participants); ?>" min="0">
    </div>
    
    <div class="td-rencontres-meta-field">
        <label for="td_nb_laboratoires">Nombre de laboratoires</label>
        <input type="number" id="td_nb_laboratoires" name="td_nb_laboratoires" value="<?php echo esc_attr($nb_laboratoires); ?>" min="0">
    </div>
    
    <?php
}

// Sauvegarder les métadonnées
function td_rencontres_save_meta($post_id) {
    if (!isset($_POST['td_rencontres_nonce']) || !wp_verify_nonce($_POST['td_rencontres_nonce'], 'td_rencontres_save_meta')) {
        return;
    }
    
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }
    
    // Sauvegarder la date de l'événement
    if (isset($_POST['td_date_evenement'])) {
        update_post_meta($post_id, '_td_date_evenement', sanitize_text_field($_POST['td_date_evenement']));
    }
    
    // Sauvegarder les dates d'inscription
    if (isset($_POST['td_date_ouverture_inscriptions'])) {
        update_post_meta($post_id, '_td_date_ouverture_inscriptions', sanitize_text_field($_POST['td_date_ouverture_inscriptions']));
    }
    
    if (isset($_POST['td_date_fermeture_inscriptions'])) {
        update_post_meta($post_id, '_td_date_fermeture_inscriptions', sanitize_text_field($_POST['td_date_fermeture_inscriptions']));
    }
    
    // Sauvegarder les autres champs
    if (isset($_POST['td_lieu'])) {
        update_post_meta($post_id, '_td_lieu', sanitize_text_field($_POST['td_lieu']));
    }
    
    if (isset($_POST['td_url_sciencesconf'])) {
        update_post_meta($post_id, '_td_url_sciencesconf', esc_url_raw($_POST['td_url_sciencesconf']));
    }
    
    if (isset($_POST['td_nb_participants'])) {
        update_post_meta($post_id, '_td_nb_participants', intval($_POST['td_nb_participants']));
    }
    
    if (isset($_POST['td_nb_laboratoires'])) {
        update_post_meta($post_id, '_td_nb_laboratoires', intval($_POST['td_nb_laboratoires']));
    }
    
    // Mettre à jour les catégories et statuts automatiquement
    td_rencontres_update_categories_and_statuts($post_id);
}
add_action('save_post_rencontre', 'td_rencontres_save_meta');



// Ajouter des classes au body selon le statut des inscriptions et de l'événement
function td_rencontres_body_class($classes) {
    if (!is_singular('rencontre')) {
        return $classes;
    }
    
    $post_id = get_the_ID();
    $terms = get_the_terms($post_id, 'categorie_rencontre');
    
    if (!$terms || is_wp_error($terms)) {
        return $classes;
    }
    
    foreach ($terms as $term) {
        switch ($term->name) {
            case 'Inscriptions ouvertes':
                $classes[] = 'inscriptions-ouvertes';
                break;
            case 'Inscriptions fermées':
                $classes[] = 'inscriptions-fermees';
                break;
            case 'À venir':
                $classes[] = 'rencontre-a-venir';
                break;
            case 'Passée':
                $classes[] = 'rencontre-passee';
                break;
        }
    }
    
    return $classes;
}
add_filter('body_class', 'td_rencontres_body_class');

// Ajouter les colonnes personnalisées dans l'admin (Mise à jour)
function td_rencontres_custom_columns($columns) {
    $new_columns = array();
    $new_columns['cb'] = $columns['cb'];
    $new_columns['title'] = $columns['title'];
    $new_columns['date_evenement'] = 'Date événement';
    $new_columns['categorie'] = 'Catégorie';
    $new_columns['lieu'] = 'Lieu';
    // Ajout de la nouvelle colonne "Lien"
    $new_columns['lien'] = 'Lien ScienceConf'; 
    // La colonne 'date' de WordPress est supprimée
    
    return $new_columns;
}
add_filter('manage_rencontre_posts_columns', 'td_rencontres_custom_columns');



// Remplir les colonnes personnalisées (Mise à jour)
function td_rencontres_custom_column_content($column, $post_id) {
    switch ($column) {
        case 'date_evenement':
            $date = get_post_meta($post_id, '_td_date_evenement', true);
            echo $date ? date_i18n('d/m/Y', strtotime($date)) : '—';
            break;
            
        case 'categorie':
            $terms = get_the_terms($post_id, 'categorie_rencontre');
            if ($terms && !is_wp_error($terms)) {
                $names = array();
                // On filtre les termes 'Inscriptions ouvertes/fermées' pour ne pas les afficher ici
                $termes_exclus = ['Inscriptions fermées', 'Inscriptions ouvertes']; 
                
                foreach ($terms as $term) {
                    if (!in_array($term->name, $termes_exclus)) {
                        $names[] = $term->name;
                    }
                }
                echo !empty($names) ? implode(', ', $names) : '—';
            } else {
                echo '—';
            }
            break;
            
        case 'lieu':
            $lieu = get_post_meta($post_id, '_td_lieu', true);
            echo $lieu ? esc_html($lieu) : '—';
            break;

        // Nouvelle colonne pour afficher le lien
        case 'lien':
            $url = get_post_meta($post_id, '_td_url_sciencesconf', true);
            if ($url) {
                // Afficher un lien cliquable dans l'admin
                echo '<a href="' . esc_url($url) . '" target="_blank">Voir le lien</a>'; 
            } else {
                echo '—';
            }
            break;
            
        // La colonne 'inscriptions' est supprimée, ainsi que son case
    }
}
add_action('manage_rencontre_posts_custom_column', 'td_rencontres_custom_column_content', 10, 2);



// Rendre les colonnes triables (Aucun changement)
function td_rencontres_sortable_columns($columns) {
    $columns['date_evenement'] = 'date_evenement';
    return $columns;
}
add_filter('manage_edit-rencontre_sortable_columns', 'td_rencontres_sortable_columns');



// Gérer le tri (Aucun changement)
function td_rencontres_column_orderby($query) {
    if (!is_admin() || !$query->is_main_query()) {
        return;
    }
    
    $orderby = $query->get('orderby');
    
    if ($orderby === 'date_evenement') {
        $query->set('meta_key', '_td_date_evenement');
        $query->set('orderby', 'meta_value');
    }
}
add_action('pre_get_posts', 'td_rencontres_column_orderby');

// Charger les blocs
require_once TD_RENCONTRES_PATH . 'blocks/liste-rencontres.php';
require_once TD_RENCONTRES_PATH . 'blocks/dernieres-passees.php';
require_once TD_RENCONTRES_PATH . 'blocks/prochaine-apercu.php';
require_once TD_RENCONTRES_PATH . 'blocks/prochaine-detail.php';