<?php
/**
 * Fonctions du thème WPCNRS
 *
 * @package WPCNRS
 */

if (!defined('ABSPATH')) {
    exit;
}

include( plugin_dir_path( __FILE__ ) . 'inc/shortcodes.php');

/**
 * Configuration du thème
 */
function td_wpcnrs_setup() {
    load_theme_textdomain('wpcnrs', get_template_directory() . '/languages');
    
    add_theme_support('title-tag');
    add_theme_support('post-thumbnails');
    add_theme_support('automatic-feed-links');
    
    add_theme_support('html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
        'style',
        'script'
    ));
    
    add_theme_support('editor-styles');
    add_editor_style('style.css');
    
    add_theme_support('align-wide');
    add_theme_support('wp-block-styles');
    add_theme_support('responsive-embeds');
    
    add_theme_support('custom-logo', array(
        'height' => 100,
        'width' => 300,
        'flex-height' => true,
        'flex-width' => true
    ));
    
    register_nav_menus(array(
        'primary' => __('Menu principal', 'wpcnrs'),
        'footer' => __('Menu pied de page', 'wpcnrs')
    ));
    
    add_image_size('wpcnrs-featured', 800, 450, true);
}
add_action('after_setup_theme', 'td_wpcnrs_setup');

/**
 * Enregistrement des zones de widgets
 */
function td_wpcnrs_widgets_init() {
    register_sidebar(array(
        'name' => __('Sidebar', 'wpcnrs'),
        'id' => 'sidebar-1',
        'description' => __('Zone de widgets pour la sidebar', 'wpcnrs'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget' => '</section>',
        'before_title' => '<h2 class="widget-title">',
        'after_title' => '</h2>'
    ));
    
    register_sidebar(array(
        'name' => __('Footer', 'wpcnrs'),
        'id' => 'footer-1',
        'description' => __('Zone de widgets pour le pied de page', 'wpcnrs'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget' => '</section>',
        'before_title' => '<h2 class="widget-title">',
        'after_title' => '</h2>'
    ));
}
add_action('widgets_init', 'td_wpcnrs_widgets_init');

/**
 * Chargement des scripts et styles
 */
function td_wpcnrs_scripts() {
    wp_enqueue_style('wpcnrs-style', get_stylesheet_uri(), array(), wp_get_theme()->get('Version'));
    
    if (is_singular() && comments_open() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
}
add_action('wp_enqueue_scripts', 'td_wpcnrs_scripts');

/**
 * Largeur du contenu
 */
if (!isset($content_width)) {
    $content_width = 1200;
}

/**
 * Affichage de la date de publication
 */
function td_wpcnrs_posted_on() {
    $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
    
    if (get_the_time('U') !== get_the_modified_time('U')) {
        $time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
    }
    
    $time_string = sprintf(
        $time_string,
        esc_attr(get_the_date(DATE_W3C)),
        esc_html(get_the_date()),
        esc_attr(get_the_modified_date(DATE_W3C)),
        esc_html(get_the_modified_date())
    );
    
    $posted_on = sprintf(
        esc_html_x('Publié le %s', 'post date', 'wpcnrs'),
        '<a href="' . esc_url(get_permalink()) . '" rel="bookmark">' . $time_string . '</a>'
    );
    
    echo '<span class="posted-on">' . $posted_on . '</span>';
}

/**
 * Affichage de l\'auteur
 */
function td_wpcnrs_posted_by() {
    $byline = sprintf(
        esc_html_x('par %s', 'post author', 'wpcnrs'),
        '<span class="author vcard"><a class="url fn n" href="' . esc_url(get_author_posts_url(get_the_author_meta('ID'))) . '">' . esc_html(get_the_author()) . '</a></span>'
    );
    
    echo '<span class="byline"> ' . $byline . '</span>';
}

/**
 * Affichage des catégories
 */
function td_wpcnrs_entry_footer() {
    if ('post' === get_post_type()) {
        $categories_list = get_the_category_list(esc_html__(', ', 'wpcnrs'));
        if ($categories_list) {
            printf('<span class="cat-links">' . esc_html__('Catégories: %1$s', 'wpcnrs') . '</span>', $categories_list);
        }
        
        $tags_list = get_the_tag_list('', esc_html_x(', ', 'list item separator', 'wpcnrs'));
        if ($tags_list) {
            printf('<span class="tags-links">' . esc_html__('Tags: %1$s', 'wpcnrs') . '</span>', $tags_list);
        }
    }
}



// Redirection automatique si une catégorie ne contient qu'un seul post
function td_redirect_single_post_in_category() {
    // On s'assure d'être sur une archive de catégorie (ou taxonomie) publique
    if (!is_category() && !is_tax()) {
        return;
    }

    // Récupère la requête principale
    global $wp_query;

    // Vérifie qu'on a exactement un seul post
    if ($wp_query->post_count === 1) {
        $post = $wp_query->posts[0];

        // Vérifie que c'est bien un post public avant de rediriger
        if (get_post_status($post->ID) === 'publish') {
            wp_safe_redirect(get_permalink($post->ID), 301);
            exit;
        }
    }
}
add_action('template_redirect', 'td_redirect_single_post_in_category');



/**
 * Ajoute la classe current-page au bouton sur les pages spécifiques
 */
function td_add_current_page_class() {
    // Vérifier si on est sur une page
    if (!is_page()) {
        return;
    }
    
    // Récupérer le slug de la page actuelle
    $current_slug = get_post_field('post_name', get_post());
    
    // Liste des slugs de pages où le script doit s'exécuter
    $allowed_pages = array('rencontres-thematiques', 'journees-scientifiques');
    
    // Vérifier si la page actuelle est dans la liste
    if (!in_array($current_slug, $allowed_pages)) {
        return;
    }
    
    // Ajouter le script JavaScript avec le slug de la page actuelle
    ?>
    <script>
    document.addEventListener("DOMContentLoaded", function() {
        // Récupérer tous les boutons
        const buttonLinks = document.querySelectorAll(".wp-block-button__link.wp-element-button");
        
        if (!buttonLinks.length) {
            return;
        }
        
        // Slug de la page actuelle
        const currentSlug = "<?php echo esc_js($current_slug); ?>";
        
        // Parcourir tous les boutons pour trouver celui qui correspond à la page actuelle
        buttonLinks.forEach(function(button) {
            const href = button.getAttribute("href");
            if (!href) return;
            
            // Vérifier si le href contient le slug de la page actuelle
            if (href.includes("/" + currentSlug) || href.endsWith(currentSlug)) {
                button.classList.add("current-page");
            }
        });
    });
    </script>
    <?php
}
add_action('wp_footer', 'td_add_current_page_class');




/**
 * Charge un fichier CSS uniquement dans le back-office WordPress
 */
function td_enqueue_admin_styles() {
    // Chemin vers le fichier CSS dans votre thème
    wp_enqueue_style(
        'td-admin-styles', // Handle unique pour votre CSS
        get_template_directory_uri() . '/css/admin-style.css', // Chemin du fichier
        array(), // Dépendances (laisser vide si aucune)
        filemtime(get_template_directory() . '/css/admin-style.css') // Version basée sur la date de modification
    );
}
add_action('admin_enqueue_scripts', 'td_enqueue_admin_styles');


/**
 * Identifie les liens externes, leur ajoute la classe 'external-link'
 * et insère directement le <span> pour l'icône.
 */
function td_add_class_to_external_links( $content ) {
    // Sécurité : ne rien faire si on est dans l'admin
    if ( is_admin() ) {
        return $content;
    }

    // Récupération du domaine du site
    $site_url = parse_url( home_url(), PHP_URL_HOST );

    // Expression régulière pour trouver tous les liens <a>
    // On capture tout le contenu du lien (sauf la balise </a>) pour l'entourer après
    // $pattern = '/(<a\s+[^>]*href=[\'"](https?:\/\/[^\'"]+))([^\'"]*)>([^<]*)<\/a>/i'; // Version 1: capture le contenu
    
    // Simplifions en se concentrant sur le href et l'ajout de classe, 
    // puis on insère le <span> juste avant la balise fermante </a>.
    // Cette regex capture la balise ouvrante complète et le contenu.
    // $1: balise ouvrante (ex: <a href="..." target="_blank">)
    // $2: l'URL (ex: https://...)
    // $3: tout ce qui est après l'URL jusqu'au > (ex: ' target="_blank">)
    // $4: le contenu du lien (ex: Mon lien)
    $pattern = '/(<a\s+[^>]*href=[\'"])(https?:\/\/[^\'"]+)([\'"][^>]*>)(.*?)(<\/a>)/i';

    $content = preg_replace_callback( $pattern, function( $matches ) use ( $site_url ) {
        $opening_tag = $matches[1] . $matches[2] . $matches[3]; // Balise ouvrante complète + attributs
        $href = $matches[2];
        $link_content = $matches[4]; // Contenu textuel/HTML actuel du lien
        $closing_tag = $matches[5]; // Balise fermante (</a>)

        // Initialisation de l'icône à insérer (vide par défaut)
        $icon_span = '';

        // Si le lien est externe
        if ( strpos( $href, $site_url ) === false ) {
            
            // 1. On s'occupe de l'ajout de la classe 'external-link'
            // Si la balise ouvrante contient déjà un attribut class="..."
            if ( preg_match( '/class=[\'"]([^\'"]+)[\'"]/', $opening_tag, $class_match ) ) {
                $classes = $class_match[1] . ' external-link';
                // On remplace l'attribut class par le nouveau
                $opening_tag = preg_replace( '/class=[\'"][^\'"]+[\'"]/', 'class=\'' . esc_attr( $classes ) . '\'', $opening_tag, 1 );
            } else {
                // Sinon on ajoute l'attribut class complet
                // On utilise un simple str_replace en se basant sur le début de la balise
                $opening_tag = str_replace( '<a ', '<a class=\'external-link\' ', $opening_tag );
            }
            
            // 2. On définit le <span> à ajouter
            // Note : L'utilisation de guillemets simples pour éviter les problèmes d'échappement
            $icon_span = '<span class=\'icon_external-link\'></span>';
        }

        // On retourne le lien modifié : balise ouvrante + contenu actuel + le <span> (si externe) + balise fermante
        return $opening_tag . $link_content . $icon_span . $closing_tag;
    }, $content );

    return $content;
}
add_filter( 'the_content', 'td_add_class_to_external_links', 20 );














function td_enqueue_laboratoire_assets() {
    // Vérifie si on est sur la page /laboratoires/
    if (is_page('laboratoires')) {

        // CSS pour le masque au hover
        wp_enqueue_style('td-laboratoire-css', get_stylesheet_directory_uri() . '/css/td-laboratoire.css', array(), '1.0');

        // JS pour le clic
        wp_enqueue_script('td-laboratoire-js', get_stylesheet_directory_uri() . '/js/td-laboratoire.js', array(), '1.0', true);
    }
}
add_action('wp_enqueue_scripts', 'td_enqueue_laboratoire_assets');
















/*
add_filter( 'render_block', function( $block_content, $block ) {

    // Vérifie qu'on cible bien le bloc du plugin Meta Field Block
    if ( isset( $block['blockName'] ) && $block['blockName'] === 'mfb/meta-field-block' ) {

        // Récupère le contenu de la valeur
        if ( preg_match( '/<div class="value">(.*?)<\/div>/', $block_content, $matches ) ) {
            $value = trim( $matches[1] );

            // Si la valeur est vide (mais pas "0"), on supprime le bloc
            if ( $value === '0' ) {
                return '';
            }
        }
    }

    return $block_content;

}, 10, 2 );
*/