<?php

// Shortcode pour afficher le lien du laboratoire
function td_lien_site_shortcode($atts) {
    // Récupérer l'ID du post courant (fonctionne dans les boucles)
    $post_id = get_the_ID();
    
    // Vérifier si nous avons un ID valide
    if (!$post_id) {
        return '';
    }
    
    // Vérifier si c'est le bon post type
    if (get_post_type($post_id) !== 'td_laboratoire') {
        return '';
    }
    
    // Récupérer la meta
    $lien_site = get_post_meta($post_id, '_td_lab_lien', true);
    
    // Retourner vide si pas de lien
    if (empty($lien_site)) {
        return '';
    }
    
    // Nettoyer et valider l'URL
    $lien_site = esc_url($lien_site);
    
    if (!$lien_site) {
        return '';
    }
    
    // Retourner le lien formaté
    return sprintf(
        '<a href="%s" target="_blank" rel="noopener noreferrer" class="td-lien-laboratoire">%s</a>',
        $lien_site,
        esc_html($lien_site)
    );
}

add_shortcode('lien_site', 'td_lien_site_shortcode');



/**
 * Crée un shortcode [bouton_lire_la_suite] pour afficher un bouton
 * qui pointe vers l'article courant dans une boucle.
 *
 * Usage : [bouton_lire_la_suite]
 * Usage avec texte personnalisé : [bouton_lire_la_suite texte="Voir le projet"]
 */
add_shortcode( 'bouton_lire_la_suite', 'creer_bouton_lien_article' );

function creer_bouton_lien_article( $atts ) {
    
    // 1. Définir les attributs par défaut
    $attributs = shortcode_atts(
        array(
            'texte' => 'Lire la suite', // Le texte par défaut du bouton
        ),
        $atts
    );

    // 2. Récupérer le lien de l'article courant dans la boucle
    $lien_article = get_permalink( get_the_ID() );

    // Si on ne trouve pas de lien, on ne retourne rien.
    if ( ! $lien_article ) {
        return '';
    }

    // 3. Construire le HTML du bouton
    // On utilise les classes CSS de base de Gutenberg pour que le bouton
    // ressemble aux autres boutons de ton site.
    $html_bouton = '<div class="wp-block-buttons">';
    $html_bouton .= '<div class="wp-block-button">';
    $html_bouton .= '<a class="wp-block-button__link" href="' . esc_url( $lien_article ) . '">';
    $html_bouton .= esc_html( $attributs['texte'] );
    $html_bouton .= '</a>';
    $html_bouton .= '</div>';
    $html_bouton .= '</div>';

    // 4. Retourner le HTML
    return $html_bouton;
}

function creer_bouton_lien_article_sciencesconf( $atts ) {
    
    // 1. Définir les attributs par défaut (le nom de la fonction a été légèrement modifié pour éviter tout conflit)
    $attributs = shortcode_atts(
        array(
            // Le texte par défaut du bouton
            'texte' => 'Inscription', 
        ),
        $atts
    );

    // Récupérer l'ID de l'article courant
    $post_id = get_the_ID();

    // 2. Récupérer le lien depuis la méta-donnée '_td_url_sciencesconf'
    // Le troisième argument 'true' assure que la valeur est retournée en chaîne de caractères (single value).
    $lien_article = get_post_meta( $post_id, '_td_url_sciencesconf', true );

    // Si on ne trouve pas de lien, on ne retourne rien.
    if ( empty( $lien_article ) ) {
        return '';
    }

    // 3. Construire le HTML du bouton
    // Ajout de 'target="_blank"' pour ouvrir le lien dans un nouvel onglet, 
    // ce qui est courant pour les liens externes.
    $html_bouton = '<div class="wp-block-buttons">';
    $html_bouton .= '<div class="wp-block-button wp-block-button_sciencesconf">';
    // Utilisez esc_url_raw() si vous êtes certain du format, mais esc_url() est plus sûr.
    $html_bouton .= '<a class="wp-block-button__link wp-block-button__link_sciencesconf external-link" href="' . esc_url( $lien_article ) . '" target="_blank" rel="noopener noreferrer">';
    $html_bouton .= esc_html( $attributs['texte'] );
    $html_bouton .= '</a>';
    $html_bouton .= '</div>';
    $html_bouton .= '</div>';

    // 4. Retourner le HTML
    return $html_bouton;
}

// Enregistrement du nouveau shortcode
// N'oubliez pas de remplacer 'bouton_sciencesconf' par le nom de shortcode que vous souhaitez utiliser
add_shortcode( 'bouton_sciencesconf', 'creer_bouton_lien_article_sciencesconf' );


/**
 * Shortcode : [liste_annees]
 * Affiche un sélecteur stylé des catégories enfants de la catégorie 'annee'
 */
function liste_annees_shortcode() {
    $parent_term = get_term_by('slug', 'annee', 'categorie_rencontre');
    if (!$parent_term || is_wp_error($parent_term)) {
        return '<p>Aucune catégorie "annee" trouvée.</p>';
    }

    $categories = get_terms(array(
        'taxonomy'   => 'categorie_rencontre',
        'hide_empty' => true,
        'parent'     => $parent_term->term_id,
        'orderby'    => 'name',
        'order'      => 'DESC'
    ));

    if (empty($categories) || is_wp_error($categories)) {
        return '<p>Aucune sous-catégorie trouvée.</p>';
    }

    $selected_slug = get_query_var('categorie_rencontre');
    if (empty($selected_slug) && isset($_GET['categorie_rencontre'])) {
        $selected_slug = sanitize_text_field($_GET['categorie_rencontre']);
    }

    ob_start();
    ?>
    <style>
        .vesica-categorias-filtro {
            display: inline-block;
            position: relative;
            margin: 1em 0;
        }
        .vesica-categorias-filtro select {
            appearance: none;
            -webkit-appearance: none;
            -moz-appearance: none;
            background-color: #fff;
            border: 1px solid #ccc;
            border-radius: 6px;
            padding: 0.5em 2.5em 0.5em 1em;
            font-size: 0.95em;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        .vesica-categorias-filtro select:hover {
            border-color: #888;
            box-shadow: 0 0 0 3px rgba(0,0,0,0.05);
        }
        .vesica-categorias-filtro::after {
            content: '▼';
            position: absolute;
            right: 1em;
            top: 50%;
            transform: translateY(-50%);
            font-size: 0.7em;
            color: #666;
            pointer-events: none;
        }
    </style>

    <div class="vesica-categorias-filtro">
        <select id="filtre_annees" onchange="if(this.value) { window.location.href = this.value; }">
            <option value="">Sélectionnez une année</option>
            <?php foreach ($categories as $cat):
                $term_link = get_term_link($cat);
                if (is_wp_error($term_link)) continue;
                $is_selected = selected($selected_slug, $cat->slug, false);
            ?>
                <option value="<?php echo esc_url($term_link); ?>" <?php echo $is_selected; ?>>
                    <?php echo esc_html($cat->name); ?>
                </option>
            <?php endforeach; ?>
        </select>
    </div>

    <script>
        document.getElementById('filtre_annees').addEventListener('change', function() {
            if (this.value === '') {
                window.location.href = '<?php echo esc_url(get_permalink()); ?>';
            }
        });
    </script>
    <?php
    return ob_get_clean();
}
add_shortcode('liste_annees', 'liste_annees_shortcode');



// Shortcode pour afficher les catégories enfants de "Type rencontre" pour le post courant
function td_categories_post_enfants_shortcode($atts) {
    global $post;

    if (!$post) {
        return '';
    }

    // Récupération du terme parent "rencontres"
    $parent_term = get_term_by('slug', 'rencontres', 'categorie_rencontre');
    if (!$parent_term) {
        return '';
    }

    // Récupération des termes du post courant
    $categories = get_the_terms($post->ID, 'categorie_rencontre');
    if (is_wp_error($categories) || empty($categories)) {
        return '';
    }

    $output = '<ul class="td-categories-list">';
    foreach ($categories as $category) {
        if ($category->parent == $parent_term->term_id) {
            $link = get_term_link($category);
            if (!is_wp_error($link)) {
                $output .= '<li><a href="' . esc_url($link) . '">' . esc_html($category->name) . '</a></li>';
            }
        }
    }
    $output .= '</ul>';

    return $output;
}
add_shortcode('td_categories_rencontres', 'td_categories_post_enfants_shortcode');




function afficher_date_inscription_shortcode( $atts ) {
    // 1. Définir les attributs par défaut, y compris un ID optionnel
    $atts = shortcode_atts(
        array(
            'id' => get_the_ID(), // Par défaut, utilise l'ID de la publication actuelle
        ),
        $atts,
        'date_inscription'
    );

    // 2. Utiliser l'ID passé en attribut, ou l'ID par défaut (qui fonctionnera en dehors de la boucle)
    $post_id = intval( $atts['id'] );

    // S'assurer que l'ID est valide
    if ( $post_id <= 0 ) {
        return '';
    }

    // 3. Récupérer la valeur de la méta-clé en utilisant l'ID explicitement
    $date_value = get_post_meta( $post_id, '_td_date_ouverture_inscriptions', true );

    // ... (Reste du code de formatage comme tu l'as défini) ...

    if ( ! empty( $date_value ) ) {
        $timestamp = strtotime( $date_value );
        
        if ( $timestamp ) {
            // Utilisation du format des réglages de WordPress (ou 'd/m/Y' si tu veux forcer)
            $date_formatee = date_i18n( get_option( 'date_format' ), $timestamp ); 
            
            return esc_html( $date_formatee );
        }
    }

    return '';
}

add_shortcode( 'date_inscription', 'afficher_date_inscription_shortcode' );
